(function () {
  let calendarDiv = null;
  let inputTarget = null;
  let today = new Date();

  // 创建日历主容器
  function createCalendar() {
    calendarDiv = document.createElement("div");
    calendarDiv.id = "calendarDiv";
    calendarDiv.style.position = "absolute";
    calendarDiv.style.zIndex = "9999";
    calendarDiv.style.border = "1px solid #ccc";
    calendarDiv.style.background = "#fff";
    calendarDiv.style.padding = "5px";
    calendarDiv.style.fontSize = "12px";
    calendarDiv.style.boxShadow = "0 2px 8px rgba(0,0,0,0.2)";
    calendarDiv.style.display = "none";
    document.body.appendChild(calendarDiv);

    // 点击外部关闭
    document.addEventListener("click", (e) => {
      if (calendarDiv.style.display === "block" && !calendarDiv.contains(e.target) && e.target !== inputTarget) {
        hideCalendar();
      }
    });

    // Esc 键关闭
    document.addEventListener("keydown", (e) => {
      if (e.key === "Escape") hideCalendar();
    });
  }

  // 渲染日历 UI
  function renderCalendar(year, month) {
    let current = new Date(year, month, 1);
    let firstDay = current.getDay();
    let daysInMonth = new Date(year, month + 1, 0).getDate();

    let html = `
      <div style="text-align:center;margin-bottom:5px;">
        <button type="button" data-action="prevYear">«</button>
        <button type="button" data-action="prevMonth">‹</button>
        <span style="margin:0 10px;font-weight:bold;">${year}年 ${month + 1}月</span>
        <button type="button" data-action="nextMonth">›</button>
        <button type="button" data-action="nextYear">»</button>
      </div>
      <table border="0" cellpadding="2" cellspacing="0" style="width:100%;text-align:center;">
        <thead>
          <tr style="background:#f0f0f0;">
            <th>日</th><th>一</th><th>二</th><th>三</th><th>四</th><th>五</th><th>六</th>
          </tr>
        </thead>
        <tbody>
    `;

    let day = 1;
    for (let i = 0; i < 6; i++) {
      html += "<tr>";
      for (let j = 0; j < 7; j++) {
        if ((i === 0 && j < firstDay) || day > daysInMonth) {
          html += "<td></td>";
        } else {
          let isToday =
            day === today.getDate() &&
            month === today.getMonth() &&
            year === today.getFullYear();

          html += `<td><span class="calendar-day" data-day="${day}" style="display:inline-block;width:22px;height:22px;line-height:22px;cursor:pointer;border-radius:4px;${isToday ? "background:#ffeb3b;font-weight:bold;" : "hover"}">${day}</span></td>`;
          day++;
        }
      }
      html += "</tr>";
    }

    html += "</tbody></table>";

    calendarDiv.innerHTML = html;

    // 绑定按钮事件
    calendarDiv.querySelectorAll("button").forEach((btn) => {
      btn.addEventListener("click", () => {
        let action = btn.getAttribute("data-action");
        if (action === "prevYear") renderCalendar(year - 1, month);
        if (action === "nextYear") renderCalendar(year + 1, month);
        if (action === "prevMonth") renderCalendar(year, month - 1);
        if (action === "nextMonth") renderCalendar(year, month + 1);
      });
    });

    // 绑定日期选择
    calendarDiv.querySelectorAll(".calendar-day").forEach((d) => {
      d.addEventListener("click", () => {
        let day = d.getAttribute("data-day");
        let mm = month + 1 < 10 ? "0" + (month + 1) : month + 1;
        let dd = day < 10 ? "0" + day : day;
        inputTarget.value = `${year}-${mm}-${dd}`;
        hideCalendar();
      });
    });
  }

  // 显示日历
  function showCalendar(input) {
    inputTarget = input;
    let rect = input.getBoundingClientRect();
    calendarDiv.style.left = rect.left + window.scrollX + "px";
    calendarDiv.style.top = rect.bottom + window.scrollY + "px";
    calendarDiv.style.display = "block";

    let now = input.value ? new Date(input.value) : new Date();
    renderCalendar(now.getFullYear(), now.getMonth());
  }

  function hideCalendar() {
    calendarDiv.style.display = "none";
  }

  // 挂载到全局
  window.EcmsSetDay = function (inputId) {
    let input = document.getElementById(inputId);
    if (!calendarDiv) createCalendar();
    showCalendar(input);
  };
})();
